% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [dthetadt] = EqnTheta(~,theta,delta_1,delta_2,alpha_psi, ...
                               Dz1,Dz2,w,theta_L,theta_R,V)
% EqnTheta time derivatives of the field at the grid points.
% Equations for the theta representation of the director field.
% 
% Input parameters:
% time      - The time when the derivatives are to be calculated
% theta     - The values of the beta representation of the tensorial 
%             director field on the interior points of a Gauss-Lobatto
%             grid.  This is a vector of length nz. 
% delta_1   - The coupling with the electro-static field.
% delta_2   - The relative difference between the two elastic constants
% alpha_psi - Electrostatic birefringence.
% Dz2       - Matrix representation of the second derivative in z.
% w         - The weights for the integral of the derivative of the
%             potential. 
% theta_L   - The boundary condition at z=-1.
% theta_R   - The boundary condition at z=+1.
% V         - The electrostatic potential
% Returns:
% dthetadt  - The time derivative of the director field at the interior 
%             points of the Gauss-Lobatto grid.  This is a vector of length
%             nz. 
    
% 1) Define a vector that has the values of theta on all the grid points.
theta_all = [theta_L ; theta ; theta_R];
%
% 2) Compute the derivative of the potential.
%
% 2.1 - Compute the normalisation coefficient of the potential.
A = V/(w*(1./(1-alpha_psi*cos(2*theta_all))));
% 2.2 - Compute the potential.
psi = A./(1-alpha_psi*cos(2*theta_all));
%
% 3) Compute the time derivative of the director field.
%
% 3.1 - Compute the first and second derivative of theta with respect to z.
dz1 = Dz1*theta_all;
dz2 = Dz2*theta_all;
% 3.2 - Compute the time derivative.
dthetadt = (1-delta_2*cos(2*theta)).*dz2(2:end-1) + ...
           delta_2*sin(2*theta).*dz1(2:end-1).^2 + ...
           delta_1*sin(2*theta).*psi(2:end-1).^2;
end